﻿using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using HIPS.Web.Components.Cache;

namespace HIPS.Web.Test.Infrastructure
{

    /// <summary>
    /// Unit tests for cache providers.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
    /// </history>
    [TestClass]
    public class CacheProviderTests : TestBase
    {

        #region Declarations

        /// <summary>
        /// Represents an item to be cached.
        /// </summary>
        /// <typeparam name="T">Type of the value to be cached.</typeparam>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
        /// </history>
        private class CacheItem<T>
        {

            #region Properties

            /// <summary>
            /// Gets or sets the value of the cached item.
            /// </summary>
            /// <history>
            ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
            /// </history>
            public T Value { get; set; }

            #endregion

            #region Constructors

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="value">Value to be cached.</param>
            /// <history>
            ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
            /// </history>
            public CacheItem(T value)
            {
                this.Value = value;
            }

            #endregion
 
        }

        #endregion

        #region Methods

        #region Tests

        /// <summary>
        /// NoCachingCacheProvider: Test that we always execute the callback method (ie, no values are cached).
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void No_Cache_Can_Get_Not_Cached()
        {
            //Arrange
            var cache = new NoCachingCacheProvider();
            string cacheKey = System.Reflection.MethodInfo.GetCurrentMethod().Name;
            var item1 = new CacheItem<System.Guid>(System.Guid.NewGuid());
            var item2 = new CacheItem<System.Guid>(System.Guid.NewGuid());

            //Act
            var result1 = cache.GetOrSet<CacheItem<System.Guid>>(cacheKey, () => { return item1; });
            var result2 = cache.GetOrSet<CacheItem<System.Guid>>(cacheKey, () => { return item2; });

            //Assert
            Assert.IsNotNull(result1, "Test for non-null result failed.");
            Assert.IsNotNull(result2, "Test for non-null result failed.");
            Assert.AreEqual(item1.Value, result1.Value, "Test for expected cached value failed.");
            Assert.AreEqual(item2.Value, result2.Value, "Test for expected cached value failed.");
            Assert.AreNotEqual(result1.Value, result2.Value, "Test for different cache values failed.");
        }

        /// <summary>
        /// MemoryCacheProvider: Test that we can cache and retrieve items from the cache.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Memory_Cache_Can_Get_Cached()
        {
            //Arrange
            var sliding = TimeSpan.FromMilliseconds(5000);
            var cache = new MemoryCacheProvider(sliding);
            string cacheKey = System.Reflection.MethodInfo.GetCurrentMethod().Name;
            var item1 = new CacheItem<System.Guid>(System.Guid.NewGuid());
            var item2 = new CacheItem<System.Guid>(System.Guid.NewGuid());

            //Act
            var result1 = cache.GetOrSet<CacheItem<System.Guid>>(cacheKey, () => { return item1; });
            var result2 = cache.GetOrSet<CacheItem<System.Guid>>(cacheKey, () => { return item2; });

            //Assert
            Assert.IsNotNull(result1, "Test for non-null result failed.");
            Assert.IsNotNull(result2, "Test for non-null result failed.");
            Assert.AreEqual(item1.Value, result1.Value, "Test for expected cached value failed.");
            Assert.AreEqual(item1.Value, result2.Value, "Test for expected cached value failed.");
            Assert.AreNotEqual(item2.Value, result2.Value, "Test for unexpected cached value failed.");
        }

        /// <summary>
        /// MemoryCacheProvider: Test that we can cache, retrieve, have expire, re-cache and re-add items in the cache.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="30 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Memory_Cache_Can_Expire_Recache()
        {
            //Arrange
            var sliding = TimeSpan.FromMilliseconds(5000);
            var cache = new MemoryCacheProvider(sliding);
            string cacheKey = System.Reflection.MethodInfo.GetCurrentMethod().Name;
            var item1 = new CacheItem<System.Guid>(System.Guid.NewGuid());
            var item2 = new CacheItem<System.Guid>(System.Guid.NewGuid());

            //Act
            var result1 = cache.GetOrSet<CacheItem<System.Guid>>(cacheKey, () => { return item1; });

            System.Threading.Thread.Sleep(sliding.Add(TimeSpan.FromMilliseconds(1000)));

            var result2 = cache.GetOrSet<CacheItem<System.Guid>>(cacheKey, () => { return item2; });

            //Assert
            Assert.IsNotNull(result1, "Test for non-null result failed.");
            Assert.IsNotNull(result2, "Test for non-null result failed.");
            Assert.AreEqual(item1.Value, result1.Value, "Test for expected cached value failed.");
            Assert.AreEqual(item2.Value, result2.Value, "Test for expected cached value failed.");
            Assert.AreNotEqual(result1.Value, result2.Value, "Test for different cache values failed.");
        }

        #endregion

        #endregion

    }
}
